//////////
//
//	File:		QTVRUtilities.c
//
//	Contains:	Some utilities for working with QuickTime and QuickTime VR movies.
//				All utilities start with the prefix "QTVRUtils_".
//
//	Written by:	Tim Monroe
//
//	Copyright:	 1996-1998 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <21>	 	02/12/98	rtm		added QTVRUtils_HideHotSpotNames and her sisters, *Show* and *Toggle*
//	   <20>	 	01/27/98	rtm		revised QTVRUtils_IsQTVRMgrInstalled and QTVRUtils_GetQTVRVersion
//	   <19>	 	01/26/98	rtm		revised QTVRUtils_GetHotSpotName to look also in hot spot atom for name atom
//	   <18>	 	01/14/98	rtm		added QTVRUtils_SetControllerType and QTVRUtils_AddStr255ToAtomContainer
//	   <17>	 	10/20/97	rtm		added QTVRUtils_IsMultiNode; added Endian*_BtoN macros to file format routines
//	   <16>	 	10/17/97	rtm		fixed QTVRUtils_IsControllerButtonVisible behavior for speaker button
//	   <15>	 	10/07/97	rtm		added cannotFindAtomErr result code to QTVRUtils_Get*AtomData functions
//	   <14>	 	09/15/97	rtm		added QTVRUtils_ToggleControllerBar
//	   <13>	 	08/21/97	rtm		added QTVRUtils_IsControllerButtonVisible
//	   <12>	 	08/19/97	rtm		added #ifdefs to support Windows compilation
//	   <11>	 	08/05/97	rtm		added QTVRUtils_GetNodeComment; still needs testing
//	   <10>	 	07/27/97	rtm		fixed QTVRUtils_GetHotSpotCount; added QTVRUtils_GetHotSpotIDByIndex
//	   <9>	 	07/25/97	rtm		revised QTVRUtils_Get*AtomData functions to use QTCopyAtomDataToPtr;
//									rewrote QTVRUtils_GetStringFromAtom
//	   <8>	 	07/24/97	rtm		removed sound volume utilities; added QTVRUtils_IsZoomAvailable;
//									revised QTVRUtils_IsQTVRMovie to use GetUserDataItem, not GetUserData
//	   <7>	 	07/23/97	rtm		revised file format utilities; added QTVRUtils_Get*AtomData functions
//	   <6>	 	07/22/97	rtm		fixed QTVRUtils_GetHotSpotCount to make sure handle is actually resized
//	   <5>	 	07/21/97	rtm		added QTVRUtils_GetNodeCount
//	   <4>	 	06/04/97	rtm		fixed QTVRUtils_ShowControllerButton and QTVRUtils_HideControllerButton,
//									and added some explanation of them; added QTVRUtils_ResetControllerButton
//	   <3>	 	02/03/97	rtm		revised QTVRUtils_ShowControllerButton and QTVRUtils_HideControllerButton 
//									to use explicit flag
//	   <2>	 	12/03/96	rtm		added controller bar utilities
//	   <1>	 	11/27/96	rtm		first file
//	   
//////////

// header files

#ifndef __QTVRUtilities__
#include "QTVRUtilities.h"
#endif

#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "QTUtilities.h"


///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// General utilities.
//
// Use these functions to get information about the availability/features of QuickTime VR or other services.
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////

//////////
//
// QTVRUtils_TrapAvailable
// Check to see whether a given trap is implemented. This is based on IM: Operating System Utilities (p. 8-22).
//
//////////

#if TARGET_OS_MAC
Boolean QTVRUtils_TrapAvailable (short theTrapWord)
{
	TrapType		myTrapType;
	short			myNumToolboxTraps;
	
	// determine whether this is a Toolbox or an Operating System trap
	if ((theTrapWord & 0x0800) > 0)
		myTrapType = ToolTrap;
	else
		myTrapType = OSTrap;

	if (myTrapType == ToolTrap) {
		theTrapWord = theTrapWord & 0x07FF;
		
		if (NGetTrapAddress(_InitGraf, ToolTrap) == NGetTrapAddress(0xAA6E, ToolTrap))
			myNumToolboxTraps = 0x0200;
		else
			myNumToolboxTraps = 0x0400;
			
		if (theTrapWord >= myNumToolboxTraps)
			theTrapWord = _Unimplemented;
	}

	return(NGetTrapAddress(theTrapWord, myTrapType) != NGetTrapAddress(_Unimplemented, ToolTrap));
}
#endif


//////////
//
// QTVRUtils_IsQTVRMgrInstalled
// Is the QuickTime VR Manager installed?
//
//////////

Boolean QTVRUtils_IsQTVRMgrInstalled (void) 
{
	Boolean 	myQTVRAvail = false;
	long		myAttrs;
	OSErr 		myErr = noErr;

	myErr = Gestalt(gestaltQTVRMgrAttr, &myAttrs);
	if (myErr == noErr)
		if (myAttrs & (1L << gestaltQTVRMgrPresent))
			myQTVRAvail = true;

	return(myQTVRAvail);
}


//////////
//
// QTVRUtils_GetQTVRVersion
// Get the version of the QuickTime VR Manager installed.
//
// The low-order word of the returned long integer contains the version number,
// so you can test a version like this:
//
//		if (QTVRUtils_GetQTVRVersion() < 0x0210)		// we require QTVR 2.1 or greater
//			return;
//
//////////

long QTVRUtils_GetQTVRVersion (void)
{
	long	 	myVersion = 0L;
	OSErr 		myErr = noErr;

	myErr = Gestalt(gestaltQTVRMgrVers, &myVersion);
	if (myErr == noErr)
		return(myVersion);
	else
		return(0L);
}


//////////
//
// QTVRUtils_GetControllerType
// Get the controller type of the specified movie.
//
//////////

OSType QTVRUtils_GetControllerType (Movie theMovie) 
{
	UserData		myUserData;
	OSType			myType = kQTVRUnknownType;
	
	// make sure we've got a movie
	if (theMovie == NULL)
		return(myType);
		
	myUserData = GetMovieUserData(theMovie);
	if (myUserData != NULL)
		GetUserDataItem(myUserData, &myType, sizeof(myType), kQTControllerType, 0);
	
	return(EndianU32_BtoN(myType));
}


//////////
//
// QTVRUtils_SetControllerType
// Set the controller type of the specified movie.
//
// This function adds an item to the movie's user data;
// the updated user data is written to the movie file when the movie is next updated
// (by calling AddMovieResource or UpdateMovieResource).
//
//////////

OSErr QTVRUtils_SetControllerType (Movie theMovie, OSType theType)
{
	UserData		myUserData;
	OSErr			myErr = noErr;

	// make sure we've got a movie
	if (theMovie == NULL)
		return(paramErr);
		
	// get the movie's user data list
	myUserData = GetMovieUserData(theMovie);
	if (myUserData == NULL)
		return(paramErr);
	
	theType = EndianU32_NtoB(theType);
	myErr = SetUserDataItem(myUserData, &theType, sizeof(theType), kUserDataMovieControllerType, 0);

	return(myErr);
}


//////////
//
// QTVRUtils_IsQTVRMovie
// Is the specified movie a QTVR movie?
//
// WARNING: This function is intended for use ONLY when you want to determine if you've got a QTVR movie
// but you don't want to use the QuickTime VR API (perhaps QTVR isn't installed...). The preferred way to
// determine if a movie is a QTVR movie is to call QTVRGetQTVRTrack and then QTVRGetQTVRInstance; if you
// get back a non-NULL instance, you've got a VR movie.
//
//////////

Boolean QTVRUtils_IsQTVRMovie (Movie theMovie) 
{
	Boolean			myIsQTVRMovie = false;
	OSType			myType;
	
	// QTVR movies have a special piece of user data identifying the movie controller type
	myType = QTVRUtils_GetControllerType(theMovie);
	
	if ((myType == kQTVRQTVRType) || (myType == kQTVROldPanoType) || (myType == kQTVROldObjectType))
		myIsQTVRMovie = true; 
		
	return(myIsQTVRMovie);
}


//////////
//
// QTVRUtils_Is20QTVRMovie
// Is the specified QTVR movie version 2.0 or greater?
//
//////////

Boolean QTVRUtils_Is20QTVRMovie (Movie theMovie) 
{
	Boolean			myIs20QTVRMovie = false;
	OSType			myType;
	
	// QTVR movies have a special piece of user data identifying the movie controller type
	myType = QTVRUtils_GetControllerType(theMovie);
	
	if (myType == kQTVRQTVRType)
		myIs20QTVRMovie = true; 
		
	return(myIs20QTVRMovie);
}


//////////
//
// QTVRUtils_IsTranslateAvailable
// Is translation currently enabled for the specified object node?
//
//////////

Boolean QTVRUtils_IsTranslateAvailable (QTVRInstance theInstance) 
{
	Boolean		myState;
	
	QTVRGetControlSetting(theInstance, kQTVRTranslation, &myState);
	return(myState);
}


//////////
//
// QTVRUtils_IsZoomAvailable
// Is zooming currently enabled for the specified object node?
//
//////////

Boolean QTVRUtils_IsZoomAvailable (QTVRInstance theInstance) 
{
	Boolean		myState;
	
	QTVRGetControlSetting(theInstance, kQTVRCanZoom, &myState);
	return(myState);
}


//////////
//
// QTVRUtils_IsPanoNode
// Is the specified node a panoramic node?
//
//////////

Boolean QTVRUtils_IsPanoNode (QTVRInstance theInstance) 
{
	return(QTVRGetNodeType(theInstance, kQTVRCurrentNode) == kQTVRPanoramaType);
}


//////////
//
// QTVRUtils_IsObjectNode
// Is the specified node an object node?
//
//////////

Boolean QTVRUtils_IsObjectNode (QTVRInstance theInstance) 
{
	return(QTVRGetNodeType(theInstance, kQTVRCurrentNode) == kQTVRObjectType);
}


//////////
//
// QTVRUtils_IsHotSpotInNode
// Does the specified node contain at least one hot spot (whether visible, enabled, or whatever)?
//
// NOTE: This is not an easy function to implement using just the QTVR 2.1 API. We do have our own
// utility QTVRUtils_GetHotSpotCount, but that function returns the number of hot spot information atoms
// in the node, which is not (necessarily) the number of hot spot regions in the hot spot image track.
// For panoramas, we could check to see if the panorama sample atom structure contains a reference
// to a hot spot image track; if it does, we'd blindly assume that that track isn't empty. For objects,
// we'll have to rely on QTVRUtils_GetHotSpotCount. So it goes....
//
// In an ideal world, there would be a hot spot information atom for each and every hot spot region in
// the hot spot image track, in which case we could be happier using QTVRUtils_GetHotSpotCount.
//
//////////

Boolean QTVRUtils_IsHotSpotInNode (QTVRInstance theInstance) 
{
	return(QTVRUtils_GetHotSpotCount(theInstance, QTVRGetCurrentNodeID(theInstance), NULL) > 0);
}


//////////
//
// QTVRUtils_IsMultiNode
// Does the specified QuickTime VR instance contain more than one node?
//
//////////

Boolean QTVRUtils_IsMultiNode (QTVRInstance theInstance) 
{
	return(QTVRUtils_GetNodeCount(theInstance) > (UInt32)1);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Controller bar utilities.
//
// Use these functions to manipulate the controller bar, its buttons, and the help text displayed in it.
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////

//////////
//
// QTVRUtils_IsControllerBarVisible
// Is the controller bar currently visible?
//
//////////

Boolean QTVRUtils_IsControllerBarVisible (MovieController theMC) 
{
	return((Boolean)MCGetVisible(theMC));
}


//////////
//
// QTVRUtils_GetControllerBarHeight
// Return the height of the controller bar displayed by the movie controller.
//
// Note that MCGetControllerBoundsRect returns rectangle of bar and movie, if attached;
// so we need to unattach the controller bar first.
//
//////////

short QTVRUtils_GetControllerBarHeight (MovieController theMC) 
{
	Boolean		wasAttached = false;
	Rect		myRect;
	short		myHeight = 0;
	
	// if the controller bar is attached, detach it (and remember we did so)
	if (MCIsControllerAttached(theMC) == 1) {
		wasAttached = true;
		MCSetControllerAttached(theMC, false);
	}
	
	// get the rectangle of the controller
	MCGetControllerBoundsRect(theMC, &myRect);
	myHeight = myRect.bottom - myRect.top;
	
	// now reattach the controller bar, if it was originally attached
	if (wasAttached)
		MCSetControllerAttached(theMC, true);
	
	return(myHeight);
}


//////////
//
// QTVRUtils_HideControllerBar
// Hide the controller bar provided by the movie controller.
//
//////////

void QTVRUtils_HideControllerBar (MovieController theMC) 
{
	MCSetVisible(theMC, false);
}


//////////
//
// QTVRUtils_ShowControllerBar
// Show the controller bar provided by the movie controller.
//
//////////

void QTVRUtils_ShowControllerBar (MovieController theMC) 
{
	MCSetVisible(theMC, true);
}


//////////
//
// QTVRUtils_ToggleControllerBar
// Toggle the state of the controller bar provided by the movie controller.
//
//////////

void QTVRUtils_ToggleControllerBar (MovieController theMC) 
{
	if (QTVRUtils_IsControllerBarVisible(theMC))
		QTVRUtils_HideControllerBar(theMC);
	else
		QTVRUtils_ShowControllerBar(theMC);
}


//////////
//
// QTVRUtils_HideControllerButton
// Hide the specified button in the controller bar.
//
// Some explanation is probably useful here: the first thing to understand is that every VR movie has 
// TWO sets of movie controller flags: (1) a set of "control flags" and (2) a set of "explicit flags".
//
// The control flags work as documented in IM: QuickTime Components (pp. 2-20f) and in VRPWQTVR2.0 (pp. 2-23f):
// if a bit in the set of control flags is set (that is, equal to 1), then the associated action or property is
// enabled. For instance, bit 17 (mcFlagQTVRSuppressZoomBtns) means to suppress the zoom buttons. So, if that
// bit is set in a VR movie's control flags, the zoom buttons are NOT displayed. If that bit is clear, the zoom
// buttons are displayed.
//
// However, the QuickTime VR movie controller sometimes suppresses buttons even when those buttons 
// have not been explicitly suppressed in the control flags. For example, if a particular VR movie does not
// contain a sound track, then the movie controller automatically suppresses the speaker/volume button. Likewise,
// if a movie does contain a sound track, then the speaker/volume button is automatically displayed, again without
// regard to the actual value of bit 17 in the control flags.
//
// This might not be what you'd like to happen. For instance, if your application is playing a sound that it
// loaded from a sound resource, you might want the user to be able to adjust the sound's volume using the volume
// control. To do that, you need a way to *force* the speaker/volume button to appear. For this reason, the
// explicit flags were introduced.
//
// The explicit flags indicate which bits in the control flags are to be used explicitly (that is, taken at
// face value). If a certain bit is set in a movie's explicit flags, then the corresponding bit in the control
// flags is interpreted as the desired setting for the feature (and the movie controller will not attempt to
// do anything "clever"). In other words, if bit 17 is set in a movie's explicit flags and bit 17 is clear in
// that movie's control flags, then the zoom buttons are displayed. Similarly, if bit 2 is set in a movie's 
// explicit flags and bit 2 is clear in that movie's control flags, then the speaker/volume button is displayed,
// whether or not the movie contains a sound track.
//
// The final thing to understand: to get or set a bit in a movie's explicit flags, you must set the flag 
// mcFlagQTVRExplicitFlagSet in your call to mcActionGetFlags or mcActionSetFlags. To get or set a bit in a 
// movie's control flags, you must clear the flag mcFlagQTVRExplicitFlagSet in your call to mcActionGetFlags 
// or mcActionSetFlags. Note that when you use the defined constants to set values in the explicit flags, the 
// constant names might be confusing. For instance, setting the bit mcFlagSuppressSpeakerButton in a movie's
// explicit flags doesn't cause the speaker to be suppressed; it just means: "use the actual value of the
// mcFlagSuppressSpeakerButton bit in the control flags".
//
// Whew! Any questions? Okay, then now you'll understand how to hide or show a button in the controller bar:
// set the appropriate explicit flag to 1 and set the corresponding control flag to the desired value. And
// you'll understand how to let the movie controller do its "clever" work: clear the appropriate explicit flag.
//
//////////

void QTVRUtils_HideControllerButton (MovieController theMC, long theButton) 
{
	long	myControllerFlags;
	
	// get the current explicit flags and set the explicit flag for the specified button
	myControllerFlags = mcFlagQTVRExplicitFlagSet;
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)((myControllerFlags | theButton) | mcFlagQTVRExplicitFlagSet));
	
	// get the current control flags and set the suppress flag for the specified button
	myControllerFlags = 0;
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)((myControllerFlags | theButton) & ~mcFlagQTVRExplicitFlagSet));
}


//////////
//
// QTVRUtils_ShowControllerButton
// Show the specified button in the controller bar.
//
//////////

void QTVRUtils_ShowControllerButton (MovieController theMC, long theButton) 
{
	long	myControllerFlags;
	
	// get the current explicit flags and set the explicit flag for the specified button
	myControllerFlags = mcFlagQTVRExplicitFlagSet;
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)((myControllerFlags | theButton) | mcFlagQTVRExplicitFlagSet));
	
	// get the current control flags and clear the suppress flag for the specified button
	myControllerFlags = 0;
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)(myControllerFlags & ~theButton & ~mcFlagQTVRExplicitFlagSet));
}


//////////
//
// QTVRUtils_ToggleControllerButton
// Toggle the state of the specified button in the controller bar.
//
//////////

void QTVRUtils_ToggleControllerButton (MovieController theMC, long theButton) 
{
	long	myControllerFlags;
	
	// get the current control flags and toggle the suppress flag for the specified button
	myControllerFlags = 0;
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	
	if (myControllerFlags & theButton)				// if the button is currently suppressed...
		QTVRUtils_ShowControllerButton(theMC, theButton);
	else
		QTVRUtils_HideControllerButton(theMC, theButton);
}


//////////
//
// QTVRUtils_ResetControllerButton
// Remove any explicit setting of the specified button in the controller bar.
// (This allows the QuickTime VR movie controller to be as clever as it knows how to be.)
//
//////////

void QTVRUtils_ResetControllerButton (MovieController theMC, long theButton) 
{
	long	myControllerFlags = mcFlagQTVRExplicitFlagSet;
	
	// get the current explicit flags and clear the explicit flag for the specified button
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)((myControllerFlags | theButton) & ~mcFlagQTVRExplicitFlagSet));
}


//////////
//
// QTVRUtils_IsControllerButtonVisible
// Is the specified button in the controller bar currently visible?
//
//////////

Boolean QTVRUtils_IsControllerButtonVisible (MovieController theMC, long theButton) 
{
	long		myControllerFlags;

	// get the current control flags
	myControllerFlags = 0;
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);

	// the speaker button requires some additional logic, because the QTVR movie controller treats it special;
	// be advised that that controller's special behavior could change in the future,
	// so you might need to tweak this code
	if (theButton == mcFlagSuppressSpeakerButton) {
		long	myExplicitFlags;
		
		// get the current explicit flags
		myExplicitFlags = mcFlagQTVRExplicitFlagSet;
		MCDoAction(theMC, mcActionGetFlags, &myExplicitFlags);
	
		// the speaker button is not showing if the movie has no sound track and the explicit flag is not set
		if (!QTUtils_MovieHasSoundTrack(MCGetMovie(theMC)) && !(myExplicitFlags & theButton))
			return(false);
	}
	
	// examine the suppress flag for the specified button
	if (myControllerFlags & theButton)				// if the button is currently suppressed...
		return(false);
	else
		return(true);
}


//////////
//
// QTVRUtils_HideHotSpotNames
// Disable the displaying of hot spot names in the controller bar.
//
//////////

void QTVRUtils_HideHotSpotNames (MovieController theMC) 
{
	QTVRUtils_HideControllerButton(theMC, kQTVRHotSpotNames);
}


//////////
//
// QTVRUtils_ShowHotSpotNames
// Enable the displaying of hot spot names in the controller bar.
//
//////////

void QTVRUtils_ShowHotSpotNames (MovieController theMC) 
{
	QTVRUtils_ShowControllerButton(theMC, kQTVRHotSpotNames);
}


//////////
//
// QTVRUtils_ToggleHotSpotNames
// Toggle the displaying of hot spot names in the controller bar.
//
//////////

void QTVRUtils_ToggleHotSpotNames (MovieController theMC) 
{
	QTVRUtils_ToggleControllerButton(theMC, kQTVRHotSpotNames);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// File format utilities.
//
// Use these functions to read information from QuickTime VR files that's not accessible using the API.
// Throughout, we assume that we're dealing with format 2.0 files. We begin with a series of functions that
// return a pointer to the data in an atom (QTVRUtils_Get*AtomData); you probably won't use these functions
// directly.
//
// Keep in mind that data stored in QuickTime atoms is big-endian. We'll need to convert any multi-byte data
// that we read from an atom to native format before we use it.
//
// Note that these file format utilities are all Getters. As yet, no Setters. Perhaps later?
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////

//////////
//
// QTVRUtils_GetVRWorldHeaderAtomData
// Get a pointer to the VR world header atom data in a QTVR movie.
//
//////////

OSErr QTVRUtils_GetVRWorldHeaderAtomData (QTVRInstance theInstance, QTVRWorldHeaderAtomPtr theVRWorldHdrAtomPtr)
{
	QTAtomContainer			myVRWorld;
	QTAtom					myAtom;
	OSErr					myErr = noErr;
		
	// get the VR world
	myErr = QTVRGetVRWorld(theInstance, &myVRWorld);
	if (myErr != noErr)
		return(myErr);
	
	// get the single VR world header atom in the VR world
	myAtom = QTFindChildByIndex(myVRWorld, kParentAtomIsContainer, kQTVRWorldHeaderAtomType, 1, NULL);
	if (myAtom != 0)
		myErr = QTCopyAtomDataToPtr(myVRWorld, myAtom, false, sizeof(QTVRWorldHeaderAtom), theVRWorldHdrAtomPtr, NULL);
	else 
		myErr = cannotFindAtomErr;
	
	QTDisposeAtomContainer(myVRWorld);
	return(myErr);
}


//////////
//
// QTVRUtils_GetNodeHeaderAtomData
// Get a pointer to the node header atom data for the node having the specified node ID.
//
//////////

OSErr QTVRUtils_GetNodeHeaderAtomData (QTVRInstance theInstance, UInt32 theNodeID, QTVRNodeHeaderAtomPtr theNodeHdrPtr)
{
	QTAtomContainer			myNodeInfo;
	QTAtom					myAtom;
	OSErr					myErr = noErr;
		
	// get the node information atom container for the specified node
	myErr = QTVRGetNodeInfo(theInstance, theNodeID, &myNodeInfo);
	if (myErr != noErr)
		return(myErr);
	
	// get the single node header atom in the node information atom container
	myAtom = QTFindChildByID(myNodeInfo, kParentAtomIsContainer, kQTVRNodeHeaderAtomType, 1, NULL);
	if (myAtom != 0)
		myErr = QTCopyAtomDataToPtr(myNodeInfo, myAtom, false, sizeof(QTVRNodeHeaderAtom), theNodeHdrPtr, NULL);
	else 
		myErr = cannotFindAtomErr;

	QTDisposeAtomContainer(myNodeInfo);
	return(myErr);
}


//////////
//
// QTVRUtils_GetHotSpotAtomData
// Get a pointer to the hot spot atom data for hot spot having the specified hot spot ID in the specified node.
//
//////////

OSErr QTVRUtils_GetHotSpotAtomData (QTVRInstance theInstance, UInt32 theNodeID, UInt32 theHotSpotID, QTVRHotSpotInfoAtomPtr theHotSpotInfoPtr)
{
	QTAtomContainer			myNodeInfo;
	QTAtom					myHSParentAtom;
	OSErr					myErr = noErr;
		
	// (1) the node information atom container contains a *hot spot parent atom*;
	// (2) the hot spot parent atom contains one or more *hot spot atoms*;
	// (3) the hot spot atom contains two children, a *general hot spot information atom*
	//     and a *specific hot spot information atom*.
	// We want to return a pointer to the general hot spot information atom data.

	// get the node information atom container for the specified node
	myErr = QTVRGetNodeInfo(theInstance, theNodeID, &myNodeInfo);
	if (myErr != noErr)
		return(myErr);
	
	// get the single hot spot parent atom in the node information atom container
	myHSParentAtom = QTFindChildByID(myNodeInfo, kParentAtomIsContainer, kQTVRHotSpotParentAtomType, 1, NULL);
	if (myHSParentAtom != 0) {
		QTAtom				myHSAtom;
		
		// get the hot spot atom whose atom ID is the specified hot spot ID
		myHSAtom = QTFindChildByID(myNodeInfo, myHSParentAtom, kQTVRHotSpotAtomType, theHotSpotID, NULL);
		if (myHSAtom != 0) {
			QTAtom			myAtom;
			
			// get the single hot spot information atom in the hot spot atom
			myAtom = QTFindChildByIndex(myNodeInfo, myHSAtom, kQTVRHotSpotInfoAtomType, 1, NULL);
			if (myAtom != 0) {
				myErr = QTCopyAtomDataToPtr(myNodeInfo, myAtom, false, sizeof(QTVRHotSpotInfoAtom), theHotSpotInfoPtr, NULL);
			}
		} else {
			myErr = cannotFindAtomErr;
		}
	} else {
		myErr = cannotFindAtomErr;
	}

	QTDisposeAtomContainer(myNodeInfo);
	return(myErr);
}


//////////
//
// QTVRUtils_GetStringFromAtom
// Get the string data from the string atom having the specified ID in the specified atom container.
//
// We use a different strategy here, since we don't know the size of the string data in advance.
//
//////////

char *QTVRUtils_GetStringFromAtom (QTAtomContainer theContainer, QTAtom theParent, QTAtomID theID)
{
	QTVRStringAtomPtr	myStringAtomPtr = NULL;
	QTAtom				myNameAtom;
	char				*myString = NULL;
 	OSErr				myErr = noErr;

	if (theContainer == NULL)
		return(myString);
		
	QTLockContainer(theContainer);
	
	myNameAtom = QTFindChildByID(theContainer, theParent, kQTVRStringAtomType, theID, NULL);
	if (myNameAtom != 0) {
		myErr = QTGetAtomDataPtr(theContainer, myNameAtom, NULL, (Ptr *)&myStringAtomPtr);
		if ((myErr == noErr) && (myStringAtomPtr != NULL)) {
			UInt16		myLength;
				
			myLength = EndianU16_BtoN(myStringAtomPtr->stringLength);
			if (myLength > 0) {
				myString = malloc(myLength + 1);
				if (myString != NULL) {
					memcpy(myString, myStringAtomPtr->theString, myLength);
					myString[myLength] = '\0';
				}
			}			
		}
	}
	
	QTUnlockContainer(theContainer);
	return(myString);
}


//////////
//
// QTVRUtils_AddStr255ToAtomContainer
// Add a Pascal string to the specified atom container; return (through theID) the ID of the new string atom.
//
//////////

OSErr QTVRUtils_AddStr255ToAtomContainer (QTAtomContainer theContainer, QTAtom theParent, Str255 theString, QTAtomID *theID)
{
	OSErr					myErr = noErr;

	*theID = 0;				// initialize the returned atom ID
	
	if ((theContainer == NULL) || (theParent == 0))
		return(paramErr);
		
	if (theString[0] != 0) {
		QTAtom				myStringAtom;
		UInt16				mySize;
		QTVRStringAtomPtr	myStringAtomPtr = NULL;
		
		mySize = sizeof(QTVRStringAtom) - 4 + theString[0] + 1;
		myStringAtomPtr = (QTVRStringAtomPtr)NewPtrClear(mySize);
		
		if (myStringAtomPtr != NULL) {
			myStringAtomPtr->stringUsage = EndianU16_NtoB(1);
			myStringAtomPtr->stringLength = EndianU16_NtoB(theString[0]);
			BlockMove(theString + 1, myStringAtomPtr->theString, theString[0]);
			myStringAtomPtr->theString[theString[0]] = '\0';
			myErr = QTInsertChild(theContainer, theParent, kQTVRStringAtomType, 0, 0, mySize, (Ptr)myStringAtomPtr, &myStringAtom);
			DisposePtr((Ptr)myStringAtomPtr);
			
			if (myErr == noErr)
				QTGetAtomTypeAndID(theContainer, myStringAtom, NULL, theID);
		}
	}
	
	return(myErr);
}


//////////
//
// QTVRUtils_GetDefaultNodeID
// Get the ID of the default node in a QTVR movie.
//
//////////

UInt32 QTVRUtils_GetDefaultNodeID (QTVRInstance theInstance)
{
	QTVRWorldHeaderAtom	 	myVRWorldHeader;
	UInt32					myNodeID = kQTVRCurrentNode;
	OSErr					myErr = noErr;
		
	myErr = QTVRUtils_GetVRWorldHeaderAtomData(theInstance, &myVRWorldHeader);
	if (myErr == noErr)
		myNodeID = EndianU32_BtoN(myVRWorldHeader.defaultNodeID);

	return(myNodeID);
}


//////////
//
// QTVRUtils_GetSceneFlags
// Get the set of flags associated with the VR scene.
// (Currently these flags are undefined, however.)
//
//////////

UInt32 QTVRUtils_GetSceneFlags (QTVRInstance theInstance)
{
	QTVRWorldHeaderAtom	 	myVRWorldHeader;
	UInt32					myFlags = 0L;
	OSErr					myErr;
		
	myErr = QTVRUtils_GetVRWorldHeaderAtomData(theInstance, &myVRWorldHeader);
	if (myErr == noErr)
		myFlags = EndianU32_BtoN(myVRWorldHeader.vrWorldFlags);

	return(myFlags);
}


//////////
//
// QTVRUtils_GetSceneName
// Get the name of the VR scene.
// The caller is responsible for disposing of the pointer returned by this function (by calling free).
//
//////////

char *QTVRUtils_GetSceneName (QTVRInstance theInstance)
{
	QTVRWorldHeaderAtom	 		myVRWorldHeader;
	char						*mySceneName = NULL;
	OSErr						myErr = noErr;
		
	myErr = QTVRUtils_GetVRWorldHeaderAtomData(theInstance, &myVRWorldHeader);
	if (myErr == noErr) {
		QTAtomID				myNameAtomID;
		
		// get the atom ID of the name string atom
		myNameAtomID = EndianU32_BtoN(myVRWorldHeader.nameAtomID);
		
		if (myNameAtomID != 0) {
			QTAtomContainer		myVRWorld;
			
			// the string atom containing the name of the scene is a *sibling* of the VR world header atom
			myErr = QTVRGetVRWorld(theInstance, &myVRWorld);
			if (myErr == noErr)
				mySceneName = QTVRUtils_GetStringFromAtom(myVRWorld, kParentAtomIsContainer, myNameAtomID);
			
			QTDisposeAtomContainer(myVRWorld);
		}
	}

	return(mySceneName);
}


//////////
//
// QTVRUtils_GetNodeCount
// Get the number of nodes in a QTVR movie.
//
//////////

UInt32 QTVRUtils_GetNodeCount (QTVRInstance theInstance)
{
	QTAtomContainer			myVRWorld;
	QTAtom					myNodeParentAtom;
	UInt32					myNumNodes = 0;
	OSErr					myErr = noErr;

	// get the VR world
	myErr = QTVRGetVRWorld(theInstance, &myVRWorld);
	if (myErr != noErr)
		return(myNumNodes);

	// get the node parent atom, whose children contain info about all nodes in the scene
	myNodeParentAtom = QTFindChildByIndex(myVRWorld, kParentAtomIsContainer, kQTVRNodeParentAtomType, 1, NULL);
	if (myNodeParentAtom != 0) {
		// now count the node ID children of the node parent atom, which is the number of nodes in the scene
		myNumNodes = QTCountChildrenOfType(myVRWorld, myNodeParentAtom, kQTVRNodeIDAtomType);
	}	

	QTDisposeAtomContainer(myVRWorld);
	
	return(myNumNodes);
}


//////////
//
// QTVRUtils_GetNodeType
// Get the type of the node with the specified ID.
//
// NOTE: This function is redundant, given QTVRGetNodeType; it's included here for illustrative purposes only.
//
//////////

OSErr QTVRUtils_GetNodeType (QTVRInstance theInstance, UInt32 theNodeID, OSType *theNodeType)
{
	QTVRNodeHeaderAtom		myNodeHeader;
	OSErr					myErr = noErr;

	// make sure we always return some meaningful value
	*theNodeType = kQTVRUnknownType;
	
	// get the node header atom data
	myErr = QTVRUtils_GetNodeHeaderAtomData(theInstance, theNodeID, &myNodeHeader);
	if (myErr == noErr)
		*theNodeType = EndianU32_BtoN(myNodeHeader.nodeType);
		
	return(myErr);
}


//////////
//
// QTVRUtils_GetNodeName
// Get the name of the node with the specified ID.
// The caller is responsible for disposing of the pointer returned by this function (by calling free).
//
//////////

char *QTVRUtils_GetNodeName (QTVRInstance theInstance, UInt32 theNodeID)
{
	QTVRNodeHeaderAtom		myNodeHeader;
	char					*myNodeName = NULL;
	OSErr					myErr = noErr;
	
	myErr = QTVRUtils_GetNodeHeaderAtomData(theInstance, theNodeID, &myNodeHeader);
	if (myErr == noErr) {
		QTAtomID				myNameAtomID;
		
		// get the atom ID of the name string atom
		myNameAtomID = EndianU32_BtoN(myNodeHeader.nameAtomID);
		
		if (myNameAtomID != 0) {
			QTAtomContainer		myNodeInfo;
			
			// the string atom containing the name of the node is a *sibling* of the node information atom
			myErr = QTVRGetNodeInfo(theInstance, theNodeID, &myNodeInfo);
			if (myErr == noErr)
				myNodeName = QTVRUtils_GetStringFromAtom(myNodeInfo, kParentAtomIsContainer, myNameAtomID);

			QTDisposeAtomContainer(myNodeInfo);
		}
	}
	
	return(myNodeName);
}


//////////
//
// QTVRUtils_GetNodeComment
// Get the comment for the node with the specified ID.
// The caller is responsible for disposing of the pointer returned by this function (by calling free).
//
//////////

char *QTVRUtils_GetNodeComment (QTVRInstance theInstance, UInt32 theNodeID)
{
	QTVRNodeHeaderAtom		myNodeHeader;
	char					*myNodeCmt = NULL;
	OSErr					myErr = noErr;
	
	myErr = QTVRUtils_GetNodeHeaderAtomData(theInstance, theNodeID, &myNodeHeader);
	if (myErr == noErr) {
		QTAtomID				myCmtAtomID;
		
		// get the atom ID of the comment string atom
		myCmtAtomID = EndianU32_BtoN(myNodeHeader.commentAtomID);
		
		if (myCmtAtomID != 0) {
			QTAtomContainer		myNodeInfo;
			
			// the string atom containing the comment for the node is a *sibling* of the node information atom
			myErr = QTVRGetNodeInfo(theInstance, theNodeID, &myNodeInfo);
			if (myErr == noErr)
				myNodeCmt = QTVRUtils_GetStringFromAtom(myNodeInfo, kParentAtomIsContainer, myCmtAtomID);

			QTDisposeAtomContainer(myNodeInfo);
		}
	}
	
	return(myNodeCmt);
}


//////////
//
// QTVRUtils_GetHotSpotCount
// Return the number of hot spots in the node with specified ID,
// and fill the specified handle with a list of the hot spot IDs.
//
// If theHotSpotIDs == NULL on entry, do not pass back the list of IDs.
//
// WARNING: This routine determines the number of hot spots by counting
// the hot spot atoms in a hot spot parent atom; this might not be
// the same as counting the number of regions in the hot spot image track.
// Sigh.
//
//////////

UInt32 QTVRUtils_GetHotSpotCount (QTVRInstance theInstance, UInt32 theNodeID, Handle theHotSpotIDs)
{
	QTAtomContainer			myNodeInfo;
	QTAtom					myHSParentAtom = 0;
	UInt32					myNumHotSpots = 0;
	OSErr					myErr = noErr;
	
	// get the node information atom container for the current node
	myErr = QTVRGetNodeInfo(theInstance, theNodeID, &myNodeInfo);
	
	// get the hot spot parent atom
	if (myErr == noErr)
		myHSParentAtom = QTFindChildByID(myNodeInfo, kParentAtomIsContainer, kQTVRHotSpotParentAtomType, 1, NULL);
		
	if (myHSParentAtom != 0) {
		SignedByte			myHState;
		Size				mySize;

		// get the number of hot spots in the current node
		myNumHotSpots = QTCountChildrenOfType(myNodeInfo, myHSParentAtom, kQTVRHotSpotAtomType);
		
		// now pass back a list of the hot spot IDs;
		// if theHotSpotIDs is NULL on entry, we assume the caller doesn't want this information
		if (theHotSpotIDs != NULL) {
		
			// unlock the handle, if it's locked (so that we can resize it)
			myHState = HGetState(theHotSpotIDs);
			if (myHState & 0x80)			// 0x80 == the block-is-locked bit in the SignedByte returned by HGetState
				HUnlock(theHotSpotIDs);

			// resize the handle to the appropriate size
			mySize = sizeof(UInt32) * myNumHotSpots;
			SetHandleSize(theHotSpotIDs, mySize);
			
			// restore the original handle state
			HSetState(theHotSpotIDs, myHState);
			
			// make sure we actually did resize the handle
			if (GetHandleSize(theHotSpotIDs) == mySize) {
				short			myIndex;
				QTAtom			myAtom;
				QTAtomID		myID;
				UInt32			*myIDPtr;
				
				myIDPtr = (UInt32 *)*theHotSpotIDs;

				// loop thru all the hot spots to get their IDs
				for (myIndex = 1; myIndex <= (short)myNumHotSpots; myIndex++) {
					myAtom = QTFindChildByIndex(myNodeInfo, myHSParentAtom, kQTVRHotSpotAtomType, myIndex, &myID);
					myIDPtr[myIndex - 1] = (UInt32)myID;
				}
			}
		}
	}
	
	QTDisposeAtomContainer(myNodeInfo);
	return(myNumHotSpots);
}


//////////
//
// QTVRUtils_GetHotSpotIDByIndex
// Return the hot spot ID having the specified index in the list of hot spot IDs returned by QTVRUtils_GetHotSpotCount,
// or kQTVRUtils_InvalidHotSpotID if no such hot spot exists.
//
//////////

UInt32 QTVRUtils_GetHotSpotIDByIndex (QTVRInstance theInstance, Handle theHotSpotIDs, UInt32 theIndex)
{
	Size			mySize;
	UInt32			myID = kQTVRUtils_InvalidHotSpotID;
	UInt32			*myIDPtr;
				
	// make sure the instance and hot spot list are non-NULL
	if ((theInstance == NULL) || (theHotSpotIDs == NULL))
		return(myID);
	
	// make sure that the index is valid	
	mySize = GetHandleSize(theHotSpotIDs);
	if (theIndex >= (mySize / sizeof(UInt32)))
		return(myID);

	myIDPtr = (UInt32 *)*theHotSpotIDs;
	myID = myIDPtr[theIndex];
	return(myID);
}


//////////
//
// QTVRUtils_GetHotSpotType
// Return the type of the hot spot having the specified hot spot ID in the specified node.
//
// NOTE: This function is semi-redundant, given QTVRGetHotSpotType; it's included here for illustrative purposes only.
// (Note, however, that QTVRGetHotSpotType returns types only for hot spots in the current node; here we do any node!)
//
//////////

OSErr QTVRUtils_GetHotSpotType (QTVRInstance theInstance, UInt32 theNodeID, UInt32 theHotSpotID, OSType *theHotSpotType)
{
	QTVRHotSpotInfoAtom		myHotSpotAtomData;
	OSErr					myErr = noErr;
	
	// make sure we always return some meaningful value
	*theHotSpotType = kQTVRHotSpotUndefinedType;
	
	// get the hot spot information atom data
	myErr = QTVRUtils_GetHotSpotAtomData(theInstance, theNodeID, theHotSpotID, &myHotSpotAtomData);
	if (myErr == noErr)
		*theHotSpotType = EndianU32_BtoN(myHotSpotAtomData.hotSpotType);		// return the hot spot type
	
	return(myErr);
}


//////////
//
// QTVRUtils_GetHotSpotName
// Return the name of the hot spot having the specified hot spot ID in the specified node.
// The caller is responsible for disposing of the pointer returned by this function (by calling free).
//
//////////

char *QTVRUtils_GetHotSpotName (QTVRInstance theInstance, UInt32 theNodeID, UInt32 theHotSpotID)
{
	QTVRHotSpotInfoAtom		myHotSpotAtomData;
	char					*myHotSpotName = NULL;
	OSErr					myErr = noErr;
	
	// get the hot spot information atom data
	myErr = QTVRUtils_GetHotSpotAtomData(theInstance, theNodeID, theHotSpotID, &myHotSpotAtomData);
	if (myErr == noErr) {
		QTAtomID				myNameAtomID;
		
		// get the atom ID of the name string atom
		myNameAtomID = EndianU32_BtoN(myHotSpotAtomData.nameAtomID);
		
		if (myNameAtomID != 0) {
			QTAtomContainer		myNodeInfo;
			QTAtom				myHSParentAtom;
			QTAtom				myHSAtom;
			QTAtom				myNameAtom = 0;
			
			// version 2.0 documentation says that the hot spot name is contained in a string atom
			// that is a sibling of the hot spot atom (that is, a child of the hot spot parent atom);
			// some other documents indicate that a string atom is always a sibling of the atom that
			// contains the reference (in this case, a sibling of the hot spot information atom, and
			// hence a child of the hot spot atom); we will look first in the hot spot atom and then
			// in the hot spot parent atom. The version 2.1 documentation corrects the earlier error.
			// Mea culpa!

			// get the hot spot parent atom and the hot spot atom
			myErr = QTVRGetNodeInfo(theInstance, theNodeID, &myNodeInfo);
			if (myErr == noErr) {
				myHSParentAtom = QTFindChildByID(myNodeInfo, kParentAtomIsContainer, kQTVRHotSpotParentAtomType, 1, NULL);
				if (myHSParentAtom != 0) {
					myHSAtom = QTFindChildByID(myNodeInfo, myHSParentAtom, kQTVRHotSpotAtomType, theHotSpotID, NULL);
					if (myHSAtom != 0) {
						QTAtom	myParentAtom;
						
						// look for a string atom that is a child of the hot spot atom
						myParentAtom = myHSAtom;
						myNameAtom = QTFindChildByID(myNodeInfo, myParentAtom, kQTVRStringAtomType, theHotSpotID, NULL);
						if (myNameAtom == 0) {
							// no such atom in the hot spot atom; look in the hot spot parent atom
							myParentAtom = myHSParentAtom;
							myNameAtom = QTFindChildByID(myNodeInfo, myParentAtom, kQTVRStringAtomType, theHotSpotID, NULL);
						}
						
						if (myNameAtom != 0)
							myHotSpotName = QTVRUtils_GetStringFromAtom(myNodeInfo, myParentAtom, myNameAtomID);
					}
				}
			}
			
			QTDisposeAtomContainer(myNodeInfo);
		}
	}

	return(myHotSpotName);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Miscellaneous utilities.
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////


//////////
//
// QTVRUtils_Point3DToPanAngle
// Return the QTVR pan angle for a given QD3D point.
// 
//////////

float QTVRUtils_Point3DToPanAngle (float theX, float theY, float theZ)
{
#pragma unused(theY)

	float	myPan;
	
	if (theZ != 0.0) {
		// note that atan always returns angles in the range -/2 to /2
		myPan = atan(theX / theZ);
		myPan = (theZ > 0) ? myPan + kVRPi : myPan;
	} else {
		myPan = (theX > 0) ? kVR3PiOver2 : kVRPiOver2;
	}
	
	// make sure myPan is positive
	while (myPan < 0.0)
		myPan += kVR2Pi;

	return(myPan);
}


//////////
//
// QTVRUtils_Point3DToTiltAngle
// Return the QTVR tilt angle for a given QD3D point.
// 
//////////

float QTVRUtils_Point3DToTiltAngle (float theX, float theY, float theZ)
{
	float			myTilt;
	float			myDistance;
	TQ3Point3D		myPoint;
	
	myPoint.x = theX;
	myPoint.y = theY;
	myPoint.z = theZ;
	
	myDistance = QTVRUtils_GetDistance(myPoint);
	if (myDistance != 0.0)
		myTilt = asin(theY / myDistance); 
	else
		myTilt = 0.0;
	
	return(myTilt);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Node callback utilities.
//
// Use these to obtain standard behaviors when entering or exiting nodes.
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////

//////////
//
// QTVRUtils_StandardEnteringNodeProc
// A standard procedure for entering a new node.
//
// This function performs actions that many applications will want done when entering a new node:
//	* display back button only if multinode movie
//	* display show-hot-spot button only if there are hotspots
//	* display the translate button only for object nodes that can translate
//	* (this space for rent)
//
//////////

PASCAL_RTN OSErr QTVRUtils_StandardEnteringNodeProc (QTVRInstance theInstance, long theNodeID, MovieController theMC)
{
#pragma unused(theNodeID)

	OSErr		myErr = noErr;

	if ((theInstance == NULL) || (theMC == NULL))
		return(paramErr);
		
	/////// 
	// all nodes
	/////// 
		
	// display the back button only if it's a multinode movie
	if (QTVRUtils_IsMultiNode(theInstance))
		QTVRUtils_ShowControllerButton(theMC, (long)mcFlagQTVRSuppressBackBtn);
	else
		QTVRUtils_HideControllerButton(theMC, (long)mcFlagQTVRSuppressBackBtn);

	// display the show-hot-spot button only if there are hotspots in the node
	if (QTVRUtils_IsHotSpotInNode(theInstance))
		QTVRUtils_ShowControllerButton(theMC, (long)mcFlagQTVRSuppressHotSpotBtn);
	else
		QTVRUtils_HideControllerButton(theMC, (long)mcFlagQTVRSuppressHotSpotBtn);

	/////// 
	// panoramic nodes
	/////// 
	
	if (QTVRUtils_IsPanoNode(theInstance)) {
	
		// hide the translate button
		QTVRUtils_HideControllerButton(theMC, (long)mcFlagQTVRSuppressTranslateBtn);
		
	} else {
	
	/////// 
	// object nodes
	/////// 
		
		// show the translate button, but only if translation is available
		if (QTVRUtils_IsTranslateAvailable(theInstance))
			QTVRUtils_ShowControllerButton(theMC, (long)mcFlagQTVRSuppressTranslateBtn);
		else
			QTVRUtils_HideControllerButton(theMC, (long)mcFlagQTVRSuppressTranslateBtn);
	}
	
	return(myErr);
}


//////////
//
// QTVRUtils_StandardLeavingNodeProc
// A standard procedure for leaving a node.
// This function performs actions that many applications will want done when leaving a node:
//	* (this space for rent)
//
// We assume that when this procedure is called, the application has decided NOT to cancel the move;
// accordingly, we always return false in theCancel.
//
//////////

PASCAL_RTN OSErr QTVRUtils_StandardLeavingNodeProc (QTVRInstance theInstance, long fromNodeID, long toNodeID, Boolean *theCancel, MovieController theMC)
{
#pragma unused(fromNodeID, toNodeID)

	OSErr		myErr = noErr;

	if ((theInstance == NULL) || (theMC == NULL))
		return(paramErr);
	
	// nothing yet....

	*theCancel = false;
	return(myErr);
}

